﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using VSI;

namespace SimpleBufferedMotionDemo
{
	public partial class Form1 : Form
	{
		StringBuilder xmlPath = new StringBuilder(@"..\..\hiconConfig.xml");

		bool MotionInProgress = false;

        bool isArmed = false;
        HiCONDLL.DeviceStatus deviceStatus = new HiCONDLL.DeviceStatus();

		public Form1()
		{
			InitializeComponent();
			HiCONDLL.vsiAPIOpenConsole();
			HiCONDLL.vsiAPIInitialize();

			HiCONDLL.vsiAPILoadXMLConfig(xmlPath);
		}
		private void btnGoOnline_Click(object sender, EventArgs e)
		{
			if (!HiCONDLL.vsiStatusIsOnline() )
				HiCONDLL.vsiAPIConnect("", 10, 5000, HiCONDLL.MANUAL_POLLING);

		}

		/* Send all axis to zero
		 * */
		private void btnGoToZero_Click(object sender, EventArgs e)
		{
			if( MotionInProgress )
				return;

			HiCONDLL.vsiCmdArm((uint)HiCONDLL.AxisMask.ALL);

			System.Threading.Thread.Sleep(1);
            HiCONDLL.vsiCmdDataExchange(ref deviceStatus);//this is needed in manual polling only
			System.Threading.Thread.Sleep(1); 

			double[] zeroPos = new double[HiCONDLL.MAX_AXIS];
			zeroPos[0] = zeroPos[1] = zeroPos[2] = zeroPos[3] = zeroPos[4] = zeroPos[5] = 0;
			HiCONDLL.vsiCmdExecuteLinearMove(HiCONDLL.AxisMask.ALL, zeroPos, 150, 5, HiCONDLL.MoveType.ABSOLUTE );
		}


	

		private void btnGO_Click(object sender, EventArgs e)
		{
			if (HiCONDLL.vsiStatusIsMoving(-1)) //axis 0 still moving
				return;

			if (!MotionInProgress)
			{
                if ( !HiCONDLL.vsiStatusIsArmed() )
                {
                    HiCONDLL.vsiCmdArm((uint)HiCONDLL.AxisMask.ALL);
                    System.Threading.Thread.Sleep(25);
                }
                
                HiCONDLL.vsiCmdSetAxisPosition(-1, 0); //set all axis to position 0

				System.Threading.Thread.Sleep(20); //wait for DLL to receive the status update from controller (only works in auto polling mode)
                HiCONDLL.vsiCmdDataExchange(ref deviceStatus);//this is needed in manual polling only
				
				HiCONDLL.vsiStatusGetAxisPositions(axisStartPositions);

				//launch buffered motion
				DoBufferedMotionSequence(60, 0.1); //max velocity, acceleration

                /*
                Action action = new Action(delegate
                    {
                        double[] positions = new double[6]{0,0,0,0,0,0};
                        HiCONDLL.vsiCmdExecuteLinearMove(HiCONDLL.AxisMask.ALL, positions, 50, 10, HiCONDLL.MoveType.ABSOLUTE);
                        Thread.Sleep(20);

                        while (HiCONDLL.vsiStatusIsMoving(-1))
                            Thread.Sleep(1);

                        Thread.Sleep(20); //wait for DLL to receive the status update from controller (only works in auto polling mode)
                        HiCONDLL.vsiCmdDataExchange();//this is needed in manual polling only

                        HiCONDLL.vsiStatusGetAxisPositions(axisStartPositions);

                        //launch buffered motion
                        DoBufferedMotionSequence(10, 1); //max velocity, acceleration
                    });
                action.BeginInvoke(null, null);*/
			}
		}


		private void timer1_Tick(object sender, EventArgs e)
		{
			string ver = "";
			HiCONDLL.vsiAPIGetVersion(ref ver);
			lblVersion.Text = "HiCON API Version: " + ver;

			OnlineLED.BackColor = HiCONDLL.vsiStatusIsOnline() ? Color.Green : Color.Red;
			MovingLED.BackColor = HiCONDLL.vsiStatusIsMoving(-1) ? Color.Green : Color.Red;

			bool armedState = false;

			if (HiCONDLL.vsiStatusIsOnline())
            {
				armedState = HiCONDLL.vsiStatusIsArmed();

				ArmLED.BackColor = armedState ? Color.Green : Color.Red;

                if (armedState != isArmed) //Arm state changed
                {
                    if (!armedState) //disarmed somehow
                    {
                        string error;
						HiCONDLL.vsiAPIGetLastNotification(out error);
                        txtError.Text = error;
                    }
                    else
						txtError.Text = "";
                }

                HiCONDLL.vsiCmdDataExchange(ref deviceStatus);

				double[] axisPositions = new double[HiCONDLL.MAX_AXIS];

				HiCONDLL.vsiStatusGetAxisPositions(axisPositions);

				XPos.Text = string.Format("{0:n4}", axisPositions[0]);
				YPos.Text = string.Format("{0:n4}", axisPositions[1]);
				ZPos.Text = string.Format("{0:n4}", axisPositions[2]);
				APos.Text = string.Format("{0:n4}", axisPositions[3]);
				BPos.Text = string.Format("{0:n4}", axisPositions[4]);
				CPos.Text = string.Format("{0:n4}", axisPositions[5]);

				//show some diagnostics 
				double[] systemVars = new double[10];
				HiCONDLL.vsiAPIGetSysVars(systemVars);

				textBox1.Text = systemVars[0].ToString(); //FifoVectorLevel milliseconds
				textBox2.Text = systemVars[1].ToString(); //CmdFIFOLevel 
				textBox3.Text = systemVars[2].ToString(); //CmdFIFOSize
				textBox4.Text = systemVars[3].ToString(); //Elapsed millisec between buffer download
				textBox5.Text = systemVars[4].ToString(); //vectors downloaded in one cycle
				textBox6.Text = systemVars[5].ToString(); //DLL internal buffer current fill level
				textBox7.Text = systemVars[6].ToString(); //DLL internal buffer max size

                btnGO.Enabled = !HiCONDLL.vsiStatusIsMoving(-1) && !MotionInProgress;

				// Controller is enabled.  perform any motion command
				if (isArmed && MotionInProgress)
				{
					GenerateMotionBuffer(); 
				}
            }
			else
				ArmLED.BackColor = Color.Red;
			
			isArmed = armedState;
		}

		//motion stuff

        double acceleration;
        double maxVelocity;
        double accelTime;

        double velocity = 0;

        double[,] buffer;
        
		volatile int MaxPointsDownload = 60; //download 60 points if fifo level fall below 150
        volatile int  MotionRunTime = 50000;
		volatile int currMotionMilliseconds = 0;
        double Multiplier = -1;

        double[] axisStartPositions = new double[HiCONDLL.MAX_AXIS];

        public void DoBufferedMotionSequence(double maxVelocity, double acceleration)
        {
            //convert values to units/ms
            this.maxVelocity = maxVelocity / 60000;
            this.acceleration = acceleration / 1000000;

            buffer = new double[MaxPointsDownload, HiCONDLL.MAX_AXIS];
            this.velocity = this.accelTime = 0;
			MotionInProgress = true;
			currMotionMilliseconds = 0;
        }

		public void GenerateMotionBuffer() //called by form timer
		{

			int currentFillLevel = 0;
			int maxFillLevel = 0;

			HiCONDLL.vsiStatusGetMotionBufferFillLevel(ref currentFillLevel, ref maxFillLevel);

			if (currentFillLevel > 200) //if buffer level is below 200ms, create new motion and download the buffer
				return;

			double counter = 0;
			int downLoadCount = 0;

            if (currMotionMilliseconds < MotionRunTime) //not done yet
            {
                for (int vector = 0; vector < MaxPointsDownload; vector++)
                {
                    if (MotionRunTime - currMotionMilliseconds <= accelTime) //apply deceleration
                    {
                        velocity -= acceleration;
                        if (velocity < 0)
                            velocity = 0;
                    }
                    else if (velocity < maxVelocity) //apply acceleration
                    {
                        velocity += acceleration;
                        if (velocity >= maxVelocity)
                            velocity = maxVelocity;

                        if (currMotionMilliseconds >= (MotionRunTime / 2) || velocity == maxVelocity)
                        {
                            this.accelTime = this.velocity / this.acceleration;
                        }
                    }

                    counter += velocity;

                    for (int axis = 0; axis < HiCONDLL.MAX_AXIS; axis++)
                    //for (int axis = 0; axis < 1; axis++)
                    {
                        buffer[vector, axis] = axisStartPositions[axis] + (counter*Multiplier);
                    }

                    downLoadCount++;
                    currMotionMilliseconds++;
                }

                //remember the last position we generated
                for (int axis = 0; axis < HiCONDLL.MAX_AXIS; axis++)
                {
                    axisStartPositions[axis] = buffer[MaxPointsDownload - 1, axis];
                }

                //give the vectors to DLL for download
                HiCONDLL.ERROR error = HiCONDLL.vsiCmdLoadBufferedMotion(buffer, downLoadCount);

                if (error != HiCONDLL.ERROR.NONE) //error occured
                {
                    string errorMessage;
                    HiCONDLL.vsiAPIGetLastNotification(out errorMessage);
                    txtError.Text = errorMessage;
                    MotionInProgress = false;
                }

            }
            else
            {
                MotionInProgress = false;
                //currMotionMilliseconds = 0;
                //Multiplier *= -1;
            }

		}

	}
}
