﻿'''///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
' HiCON Motion API Sample C# Project
' Copyright Vital Systems Inc. 2014
'''///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Imports System
Imports System.Collections.Generic
Imports System.ComponentModel
Imports System.Data
Imports System.Drawing
Imports System.Linq
Imports System.Text
Imports System.Windows.Forms





Namespace VSI
    Partial Public Class Form1
        Inherits Form
        Private xmlPath As String = "..\..\hiconConfig.xml"

        Private digitalInputs As PictureBox(,) = New PictureBox(3, 15) {}
        Private digitalOutputs As PictureBox(,) = New PictureBox(3, 7) {}

        Private controllerOnline As Boolean = False
        Private controllerArmed As Boolean = False

        Private controllerFollowingErrorBits As UInteger = 0
        Private controllerMessage As String = ""

        Private controllerAxisPositions As Double() = New Double(HiCONDLL.MAX_AXIS - 1) {}
        Private controllerMoveSeqIDs As UInteger() = New UInteger(HiCONDLL.MAX_AXIS - 1) {}
        Private controllerAxisHomed As Boolean() = New Boolean(HiCONDLL.MAX_AXIS - 1) {}
        Private controllerAxisMoving As Boolean() = New Boolean(HiCONDLL.MAX_AXIS - 1) {}

        Private controllerDACVoltage As Double() = New Double(HiCONDLL.MAX_DAC_CHANNELS - 1) {}

        Private motionSequence As MotionSequence = Nothing


        Public Sub New()
            HiCONDLL.vsiAPIInitialize()
            HiCONDLL.vsiAPIOpenConsole()

            InitializeComponent()
            InitializeIO(digitalInputGroup, digitalInputs)
            InitializeIO(digitalOutputGroup, digitalOutputs)

            cbAxis.SelectedIndex = 0

            Dim dac As Integer = 0
            While dac < HiCONDLL.MAX_DAC_CHANNELS
                cbDACChannel.Items.Add(dac)
                System.Math.Max(System.Threading.Interlocked.Increment(dac), dac - 1)
            End While

            cbDACChannel.SelectedIndex = 0

            txtAccel.Text = "10.00"
            txtPosition.Text = "30.00"
            txtVelocity.Text = "100.00"

            XPos.Tag = 0
            YPos.Tag = 1
            ZPos.Tag = 2
            APos.Tag = 3
            BPos.Tag = 4


            CPos.Tag = 5
        End Sub


        Protected Overrides Sub OnClosing(e As CancelEventArgs)
            HiCONDLL.vsiCmdDisarm()
            HiCONDLL.vsiAPIDisconnect()

            MyBase.OnClosing(e)
        End Sub


        ''' <summary>
        ''' Organize the I/O LED pictureboxes into a 2D array for easy reference
        ''' </summary>
        Private Sub InitializeIO(groupBox As GroupBox, array As PictureBox(,))
            Dim loadOutputs As Boolean = If((array Is digitalOutputs), True, False)
            Dim maxPins As Integer = If((array Is digitalOutputs), 8, 16)

            Dim port As UShort = 0
            While port < 4
                Dim portLabel As New Label()
                portLabel.Text = "P" + (port + 11).ToString()
                portLabel.Width = 30
                portLabel.Location = New Point(0, port * 25) + New Size(10, 42)

                portLabel.Parent = groupBox

                Dim pin As UShort = 0
                While pin < maxPins
                    Dim pictureBox As New PictureBox()
                    pictureBox.BackColor = Color.LightGray
                    pictureBox.BorderStyle = BorderStyle.Fixed3D
                    pictureBox.Size = New Size(12, 20)
                    pictureBox.Location = New Point(pin * 13, port * 25) + New Size(45, 40)
                    pictureBox.Parent = groupBox

                    Dim tag As UShort = (((port + 11) << 8) Or pin)
                    pictureBox.Tag = tag
                    array(port, pin) = pictureBox

                    If loadOutputs Then
                        AddHandler pictureBox.Click, AddressOf OutputLED_Clicked
                        pictureBox.Cursor = Cursors.Hand
                    End If

                    If port = 0 AndAlso pin Mod 4 = 0 Then
                        Dim pinLabel As New Label()
                        pinLabel.Text = pin.ToString()
                        pinLabel.Size = New Size(20, 18)
                        pinLabel.Location = New Point(pin * 13, 0) + New Size(43, 22)
                        pinLabel.Parent = groupBox
                    End If
                    pin = pin + 1
                End While
                port = port + 1
            End While

            groupBox.AutoSize = True
            groupBox.Refresh()
        End Sub


        ''' <summary>
        ''' Toggle the output LED when clicked
        ''' </summary>
        Private Sub OutputLED_Clicked(sender As Object, e As EventArgs)
            Dim pictureBox As PictureBox = DirectCast(sender, PictureBox)

            Dim tag As UShort = DirectCast(pictureBox.Tag, UShort)
            Dim port As Integer = tag >> 8
            Dim pinNumber As Integer = tag And &HFF

            Dim newPinState As Boolean = pictureBox.BackColor = Color.Red
            HiCONDLL.vsiCmdSetDigitalOutput(port, pinNumber, newPinState)
        End Sub


        Private Sub OnSuccessfulConnect()
            Dim serial As New String(New Char(9) {})
            HiCONDLL.vsiStatusGetSerial(serial)

            serialTxtBox.Text = serial.Trim()
            serialTxtBox.Enabled = False

            For Each pictureBox As PictureBox In digitalInputs
                pictureBox.BackColor = Color.Red
            Next
            For Each pictureBox As PictureBox In digitalOutputs
                pictureBox.BackColor = Color.Red
            Next
        End Sub



        Private Sub EnableMotionControl(enable As Boolean)
            moveButton.Enabled = enable
            homeButton.Enabled = enable
            zeroAllButton.Enabled = enable

            executeTestButton.Enabled = enable

            cancelMoveButton.Enabled = Not enable
            cancelAllButton.Enabled = Not enable
            clearPositionButton.Enabled = Not enable
        End Sub


        Private Sub DisableDeviceControl(enable As Boolean)
            FollowErrorLED.BackColor = Color.LightGray
            MotionDoneLED.BackColor = Color.LightGray
            btnARMPID.BackColor = Color.LightGray
            picHomeStatus.BackColor = Color.LightGray

            btnARMPID.BackColor = Color.LightGray
            btnARMPID.Text = If(controllerArmed, "DISARM", "ARM")
            btnConnect.Text = If(controllerOnline, "DISCONNECT", "CONNECT")

            serialTxtBox.Enabled = True

            For Each pictureBox As PictureBox In digitalInputs
                pictureBox.BackColor = Color.LightGray
            Next
            For Each pictureBox As PictureBox In digitalOutputs
                pictureBox.BackColor = Color.LightGray
            Next
        End Sub

        Private Sub DisplayLastError()
            'HiCONDLL.vsiAPIGetLastNotification(ref controllerMessage);
            MsgDisp.Text = controllerMessage.Trim()
        End Sub


        Private Sub OnControlTimerTick(sender As Object, e As EventArgs) Handles controlTimer.Tick
            controllerOnline = HiCONDLL.vsiStatusIsOnline()

            If Not controllerOnline Then
                Return
            End If

            'required to exchange data and maintain connection with the device.
            HiCONDLL.vsiCmdDataExchange()


            Dim axis As Integer = 0
            While axis < HiCONDLL.MAX_AXIS
                controllerAxisMoving(axis) = HiCONDLL.vsiStatusIsMoving(axis)
                controllerAxisHomed(axis) = HiCONDLL.vsiStatusIsHomeFound(axis)
                System.Math.Max(System.Threading.Interlocked.Increment(axis), axis - 1)
            End While

            HiCONDLL.vsiStatusGetFollowErrorBits(controllerFollowingErrorBits)
            HiCONDLL.vsiStatusGetAxisPositions(controllerAxisPositions)
        End Sub


        Private Sub OnGUITimerTick(sender As Object, e As EventArgs) Handles guiTimer.Tick
            Dim isArmed As Boolean = HiCONDLL.vsiStatusIsArmed()

            If isArmed <> controllerArmed Then
                If Not isArmed Then
                    HiCONDLL.vsiAPIGetLastNotification(controllerMessage)
                Else
                    controllerMessage = ""
                End If

                MsgDisp.Text = controllerMessage.Trim()
                controllerArmed = isArmed
            End If

            If Not (motionSequence Is Nothing) AndAlso motionSequence.Complete Then
                MsgDisp.Text = motionSequence.ErrorMessage
                motionSequence.Cancel()
                motionSequence = Nothing
            End If


            If Not (motionSequence Is Nothing) AndAlso motionSequence.Complete Then
                MsgDisp.Text = motionSequence.ErrorMessage
                motionSequence.Cancel()
                motionSequence = Nothing
            End If


            If Not controllerOnline Then
                DisableDeviceControl(False)
                Return
            End If

            btnARMPID.BackColor = If((controllerArmed), Color.Lime, Color.Red)
            btnARMPID.Text = If((controllerArmed), "DISARM", "ARM")

            FollowErrorLED.BackColor = If((controllerFollowingErrorBits <> 0), Color.Red, Color.Lime)
            MotionDoneLED.BackColor = If(controllerAxisMoving(cbAxis.SelectedIndex), Color.Red, Color.Lime)

            picHomeStatus.BackColor = If(controllerAxisHomed(cbAxis.SelectedIndex), Color.LightGreen, Color.Red)

            Dim newPinState As Boolean = False
            Dim port As Integer = 11
            While port <= 14
                Dim pin As Integer = 0
                While pin < 16
                    'Read the digital input status
                    newPinState = HiCONDLL.vsiStatusGetDigitalInput(port, pin)
                    digitalInputs(port - 11, pin).BackColor = If(newPinState, Color.Lime, Color.Red)
                    pin = pin + 1
                End While

                pin = 0
                While pin < 8
                    'Read the digital output status
                    newPinState = HiCONDLL.vsiStatusGetDigitalOutput(port, pin)
                    digitalOutputs(port - 11, pin).BackColor = If(newPinState, Color.Lime, Color.Red)
                    pin = pin + 1
                End While
                port = port + 1
            End While

            EnableMotionControl(Not controllerAxisMoving(cbAxis.SelectedIndex))

            XPos.Text = String.Format("{0:n4}", controllerAxisPositions(0))
            YPos.Text = String.Format("{0:n4}", controllerAxisPositions(1))
            ZPos.Text = String.Format("{0:n4}", controllerAxisPositions(2))
            APos.Text = String.Format("{0:n4}", controllerAxisPositions(3))
            BPos.Text = String.Format("{0:n4}", controllerAxisPositions(4))
            CPos.Text = String.Format("{0:n4}", controllerAxisPositions(5))

            Dim systemVars As Double() = New Double(9) {}
            HiCONDLL.vsiAPIGetSysVars(systemVars)

            textBox1.Text = systemVars(0).ToString()
            'FifoVectorLevel milliseconds
            textBox2.Text = systemVars(1).ToString()
            'CmdFIFOLevel 
            textBox3.Text = systemVars(2).ToString()
            'CmdFIFOSize
            textBox4.Text = systemVars(3).ToString()
            'Elapsed millisec between buffer download
            textBox5.Text = systemVars(4).ToString()
            'vectors downloaded in one cycle

        End Sub


        Private Sub btnConnect_Click(sender As Object, e As EventArgs) Handles btnConnect.Click
            If controllerOnline Then
                HiCONDLL.vsiAPIDisconnect()
                Return
            End If

            HiCONDLL.vsiAPILoadXMLConfig(xmlPath)
            Dim serial As String = serialTxtBox.Text.Trim()

            If HiCONDLL.vsiAPIConnect(serial, controlTimer.Interval, 2000, False) <> HiCONDLL.[ERROR].NONE Then
                HiCONDLL.vsiAPIGetLastNotification(controllerMessage)
                MessageBox.Show("Connection Failed: " + controllerMessage, "HiCON CSharp Demo")
                Return
            Else
                OnSuccessfulConnect()
            End If
        End Sub


        Private Sub btnARMPID_Click(sender As Object, e As EventArgs) Handles btnARMPID.Click
            If controllerArmed Then
                HiCONDLL.vsiCmdDisarm()
            Else
                HiCONDLL.vsiCmdArm(HiCONDLL.AxisMask.X Or HiCONDLL.AxisMask.Y Or HiCONDLL.AxisMask.Z Or HiCONDLL.AxisMask.A Or HiCONDLL.AxisMask.B Or HiCONDLL.AxisMask.C)
            End If
        End Sub


        Private Sub btnMove_Click(sender As Object, e As EventArgs) Handles moveButton.Click
            'double[] positions = new double[HiCONDLL.MAX_AXIS];
            '            Array.Copy(axisPositions, positions, HiCONDLL.MAX_AXIS);
            '
            '            positions[0] = 1;
            '            positions[1] = 5;
            '            positions[3] = 10;
            '
            '            HiCONDLL.vsiCmdExecuteLinearMove(11, positions, 100, 10, HiCONDLL.VSI_MOVE_ABSOLUTE);
            '            


            Dim moveFlags As HiCONDLL.MoveType = 0

            If relativePositionBtn.Checked Then
                moveFlags = HiCONDLL.MoveType.RELATIVE
            ElseIf absolutePositionBtn.Checked Then
                moveFlags = HiCONDLL.MoveType.ABSOLUTE
            ElseIf velocityMotionBtn.Checked Then
                moveFlags = HiCONDLL.MoveType.VELOCITY
            End If

            If HiCONDLL.vsiCmdExecuteMove(cbAxis.SelectedIndex,
                Convert.ToDouble(txtPosition.Text),
                Convert.ToDouble(txtVelocity.Text),
                Convert.ToDouble(txtAccel.Text),
                moveFlags,
                (controllerMoveSeqIDs(cbAxis.SelectedIndex) = controllerMoveSeqIDs(cbAxis.SelectedIndex) + 1) <> HiCONDLL.[ERROR].NONE) Then
                DisplayLastError()
            End If
        End Sub

        Private Sub zeroMoveBtn_Click(sender As Object, e As EventArgs) Handles zeroAllButton.Click
            Dim axis As Integer = 0
            While axis < HiCONDLL.MAX_AXIS
                For Each control As Control In axisGroup.Controls
                    If Not (control.Tag Is Nothing) AndAlso DirectCast(control.Tag, Integer) = axis Then
                        Dim label As Label = DirectCast(control, Label)
                        If Math.Abs(Convert.ToSingle(label.Text)) < 0.0001F Then
                            GoTo SKIP
                        End If
                    End If
                Next

                If HiCONDLL.vsiCmdExecuteMove(cbAxis.SelectedIndex, 0, Convert.ToDouble(txtVelocity.Text), Convert.ToDouble(txtAccel.Text), HiCONDLL.MoveType.ABSOLUTE, (controllerMoveSeqIDs(cbAxis.SelectedIndex) = controllerMoveSeqIDs(cbAxis.SelectedIndex) + 1)) <> HiCONDLL.[ERROR].NONE Then
                    DisplayLastError()
                End If
SKIP:

                Continue While
                System.Math.Max(System.Threading.Interlocked.Increment(axis), axis - 1)
            End While
        End Sub


        Private Sub btnCancelMove_Click(sender As Object, e As EventArgs) Handles cancelMoveButton.Click
            motionSequence.Cancel()

            If HiCONDLL.vsiCmdCancelMove(cbAxis.SelectedIndex, False) <> HiCONDLL.[ERROR].NONE Then
                DisplayLastError()
            End If
        End Sub


        Private Sub cancelAllBtn_Click(sender As Object, e As EventArgs) Handles cancelAllButton.Click
            motionSequence.Cancel()

            If HiCONDLL.vsiCmdCancelMove(-1, False) <> HiCONDLL.[ERROR].NONE Then
                DisplayLastError()
            End If
        End Sub

        Private Sub btnClearAxisPosition_Click(sender As Object, e As EventArgs) Handles clearPositionButton.Click
            If HiCONDLL.vsiCmdClearAxisPosition(cbAxis.SelectedIndex) <> HiCONDLL.[ERROR].NONE Then
                DisplayLastError()
            End If

            Dim pos As Integer = 0
            Dim buffer As Double(,) = New Double(4999, HiCONDLL.MAX_AXIS - 1) {}

            Dim vector As Integer = 0
            While vector < 5000
                Dim axis As Integer = 0
                While axis < HiCONDLL.MAX_AXIS
                    buffer(vector, axis) = pos
                    System.Math.Max(System.Threading.Interlocked.Increment(axis), axis - 1)
                End While

                System.Math.Max(System.Threading.Interlocked.Increment(pos), pos - 1)
                System.Math.Max(System.Threading.Interlocked.Increment(vector), vector - 1)
            End While

            HiCONDLL.vsiCmdLoadBufferedMotion(buffer, 5000)
        End Sub


        Private Sub btnHome_Click(sender As Object, e As EventArgs) Handles homeButton.Click
            If HiCONDLL.vsiCmdExecuteHoming(cbAxis.SelectedIndex, 0, 30.0, 7.5) <> HiCONDLL.[ERROR].NONE Then
                DisplayLastError()
            End If
        End Sub


        Private Sub velocityChkBox_CheckedChanged(sender As Object, e As EventArgs)

        End Sub


        Private Sub jogLeftButton_MouseDown(sender As Object, e As MouseEventArgs) Handles jogLeftButton.MouseDown
            HiCONDLL.vsiCmdExecuteMove(cbAxis.SelectedIndex, -9999, Convert.ToDouble(txtVelocity.Text), Convert.ToDouble(txtAccel.Text), HiCONDLL.MoveType.ABSOLUTE, 9999)
        End Sub


        Private Sub jogRightButton_MouseDown(sender As Object, e As MouseEventArgs) Handles jogRightButton.MouseDown
            HiCONDLL.vsiCmdExecuteMove(cbAxis.SelectedIndex, 9999, Convert.ToDouble(txtVelocity.Text), Convert.ToDouble(txtAccel.Text), HiCONDLL.MoveType.ABSOLUTE, 9999)
        End Sub


        Private Sub JogButtonRelease(sender As Object, e As MouseEventArgs) Handles jogRightButton.MouseUp, jogLeftButton.MouseUp
            Dim sequenceID As UInteger = 0
            If HiCONDLL.vsiStatusGetMotionSequenceID(cbAxis.SelectedIndex, sequenceID) <> HiCONDLL.[ERROR].NONE Then
                Return
            End If

            If sequenceID <> 9999 Then
                Return
            End If

            If HiCONDLL.vsiCmdCancelMove(cbAxis.SelectedIndex, False) <> HiCONDLL.[ERROR].NONE Then
                DisplayLastError()
            End If
        End Sub




        Private Sub OnExecuteMotionSequenceButton_Click(sender As Object, e As EventArgs) Handles executeTestButton.Click
            If Not (motionSequence Is Nothing) Then
                motionSequence.Cancel()
            End If

            motionSequence = New ArcMotionSequence(Convert.ToDouble(txtVelocity.Text), Convert.ToDouble(txtAccel.Text))
            'motionSequence = new LinearMotionSequence(Convert.ToDouble(txtVelocity.Text), Convert.ToDouble(txtAccel.Text));

            motionSequence.ExecuteAsync()
        End Sub


        Private Sub OnDacChannelChanged(sender As Object, e As EventArgs) Handles cbDACChannel.SelectedIndexChanged
            sliderDACVoltage.Value = (controllerDACVoltage(cbDACChannel.SelectedIndex) * 1000)
        End Sub


        Private Sub OnDACVoltageSliderValueChanged(sender As Object, e As EventArgs) Handles sliderDACVoltage.ValueChanged
            controllerDACVoltage(cbDACChannel.SelectedIndex) = Convert.ToDouble(sliderDACVoltage.Value) / 1000.0
            If HiCONDLL.vsiCmdSetDACOutput(cbDACChannel.SelectedIndex, controllerDACVoltage(cbDACChannel.SelectedIndex)) <> HiCONDLL.[ERROR].NONE Then
                DisplayLastError()
            End If

            textboxDACVolts.Text = String.Format("{0:n3}V", controllerDACVoltage(cbDACChannel.SelectedIndex))
        End Sub








    End Class
End Namespace


