﻿//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// DSPMC Motion API Sample C# Project
// Copyright Vital Systems Inc. 2014
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Text;





namespace CsharpApp
{
    public partial class Form1 : Form
    {
        public const int MAX_AXIS = 8;

        public const int AXIS_X0 = 1;
        public const int AXIS_Y1 = 2;
        public const int AXIS_Z2 = 4;
        public const int AXIS_A3 = 8;
        public const int AXIS_B4 = 16;
        public const int AXIS_C5 = 32;
        public const int AXIS_D6 = 64;
        public const int AXIS_E7 = 128;
        public const int AXIS_ALL = byte.MaxValue;



        string xmlPath = @"..\..\dspmcConfig.xml";

        bool isOnline = false;
        bool isArmed = false;
        bool isHomeFound = false;
        uint followingErrorBits = 0;

        double[] axisPositions = new double[MAX_AXIS];


        PictureBox[,] digitalInputs = new PictureBox[4, 16];
        PictureBox[,] digitalOutputs = new PictureBox[4, 8];

        UInt32[] MoveSeqID;

        double[] currentDACVolts = new double[8];


        public Form1()
        {
            

            InitializeComponent();
            InitializeIO(digitalInputGroup, digitalInputs);
            InitializeIO(digitalOutputGroup, digitalOutputs);

            cbAxis.SelectedIndex = 0;
            dacChannelCombobox.SelectedIndex = 0;

            MoveSeqID = new UInt32[MAX_AXIS];
            txtAccel.Text = "10.00";
            txtPosition.Text = "30.00";
            txtVelocity.Text = "100.00";

            XPos.Tag = 0;
            YPos.Tag = 1;
            ZPos.Tag = 2;
            APos.Tag = 3;
            BPos.Tag = 4;
            CPos.Tag = 5;

            DSPMCLib.vsiAPIOpenConsole();
            DSPMCLib.vsiAPIInitialize();
            DSPMCLib.vsiAPILoadXMLConfig(xmlPath);
        }


        /// <summary>
        /// Organize the I/O LED pictureboxes into a 2D array for easy reference
        /// </summary>
        void InitializeIO(GroupBox groupBox, PictureBox[,] array)
        {
            bool loadOutputs = (array == digitalOutputs) ? true : false;
            int maxPins = (array == digitalOutputs) ? 8 : 16;

            for(ushort port = 0; port < 4; port++)
            {
                Label portLabel = new Label();
                portLabel.Text = "P" + (port + 11).ToString();
                portLabel.Width = 30;
                portLabel.Location = new Point(0, port * 25) + new Size(10, 42);

                portLabel.Parent = groupBox;

                for (ushort pin = 0; pin < maxPins; pin++)
                {
                    PictureBox pictureBox = new PictureBox();
                    pictureBox.BackColor = Color.LightGray;
                    pictureBox.BorderStyle = BorderStyle.Fixed3D;
                    pictureBox.Size = new Size(12, 20);
                    pictureBox.Location = new Point(pin * 13, port * 25) + new Size(45, 40);
                    pictureBox.Parent = groupBox;

                    ushort tag = (ushort)(((port + 11) << 8) | pin);
                    pictureBox.Tag = tag;
                    array[port, pin] = pictureBox;

                    if (loadOutputs)
                    {
                        pictureBox.Click += OutputLED_Clicked;
                        pictureBox.Cursor = Cursors.Hand;
                    }

                    if(port == 0 && pin % 4 == 0)
                    {
                        Label pinLabel = new Label();
                        pinLabel.Text = pin.ToString();
                        pinLabel.Size = new Size(20, 18);
                        pinLabel.Location = new Point(pin * 13, 0) + new Size(43, 22);
                        pinLabel.Parent = groupBox;
                    }
                }
            }

            groupBox.AutoSize = true;
            groupBox.Refresh();
        }


        /// <summary>
        /// Toggle the output LED when clicked
        /// </summary>
        private void OutputLED_Clicked(object sender, EventArgs e)
        {
            PictureBox pictureBox = (PictureBox)sender;

            ushort tag = (ushort)pictureBox.Tag;
            int port = tag >> 8;
            int pinNumber = tag & 0xff;

            bool newPinState = pictureBox.BackColor == Color.Red;
            DSPMCLib.vsiCmdSetDigitalOutput(port, pinNumber, newPinState);
        }


        void OnSuccessfulConnect()
        {
            string serial = new string(new char[10]);
            DSPMCLib.vsiStatusGetSerial(ref serial);

            serialTxtBox.Text = serial.Trim();
            serialTxtBox.Enabled = false;

            foreach (PictureBox pictureBox in digitalInputs)
                pictureBox.BackColor = Color.Red;
            foreach (PictureBox pictureBox in digitalOutputs)
                pictureBox.BackColor = Color.Red;
        }


        protected override void OnFormClosed(FormClosedEventArgs e)
        {
            DSPMCLib.vsiAPIDispose();
        }


        void EnableMotionControl(bool enable)
        {
            moveButton.Enabled = enable;
            homeButton.Enabled = enable;
            zeroAllButton.Enabled = enable;

            cancelMoveButton.Enabled = !enable;
            cancelAllButton.Enabled = !enable;
            clearPositionButton.Enabled = !isArmed;
        }


        void DeactivateFormControls()
        {
            btnConnect.Enabled = !isOnline;
            FollowErrorLED.BackColor = Color.LightGray;
            MotionDoneLED.BackColor = Color.LightGray;
            btnARMPID.BackColor = Color.LightGray;
            picHomeStatus.BackColor = Color.LightGray;

            btnARMPID.BackColor = Color.LightGray;
            btnARMPID.Text = isArmed ? "DISARM" : "ARM";

            serialTxtBox.Enabled = true;

            foreach (PictureBox pictureBox in digitalInputs)
                pictureBox.BackColor = Color.LightGray;
            foreach (PictureBox pictureBox in digitalOutputs)
                pictureBox.BackColor = Color.LightGray;
        }


        private void DisplayLastError()
        {
            string error = new string(new char[256]);
            int length = 0;
            DSPMCLib.vsiAPIGetLastNotification(ref error, ref length);
            MsgDisp.Text = error.Trim();
        }


        private void timer1_Tick(object sender, EventArgs e)
        {
            DSPMCLib.vsiStatusIsOnline(ref isOnline);
            btnConnect.Enabled = !isOnline;

            if (isOnline)
            {
                bool armedState = false;
                DSPMCLib.vsiStatusIsArmed(ref armedState);
                DSPMCLib.vsiStatusGetFollowErrorBits(ref followingErrorBits);

                
              

                    DSPMCLib.vsiCmdDataExchange();

                if (armedState != isArmed)
                {
                    string error = new string(new char[256]);
                    int length = 0;
                    DSPMCLib.vsiAPIGetLastNotification(ref error, ref length);
                    MsgDisp.Text = error.Trim();
                }

                bool axisMoving = false;
                DSPMCLib.vsiStatusIsMoving(cbAxis.SelectedIndex, ref axisMoving);
                DSPMCLib.vsiStatusIsHomeFound(cbAxis.SelectedIndex, ref isHomeFound);

                FollowErrorLED.BackColor = (followingErrorBits != 0) ? Color.Red : Color.Lime;
                MotionDoneLED.BackColor = (axisMoving) ? Color.Red : Color.Lime;
                btnARMPID.BackColor = (isArmed) ? Color.Lime : Color.Red;
                btnARMPID.Text = (isArmed) ? "DISARM" : "ARM";

                picHomeStatus.BackColor = isHomeFound ? Color.LightGreen : Color.Red;

                bool newPinState = false;
                for (int port = 11; port <= 14; port++)
                {
                    for (int pin = 0; pin < 16; pin++) //Read the digital input status
                    {
                        DSPMCLib.vsiStatusGetDigitalInput(port, pin, ref newPinState);
                        digitalInputs[port - 11, pin].BackColor = newPinState ? Color.Lime : Color.Red;
                    }

                    for (int pin = 0; pin < 8; pin++) //Read the digital output status
                    {
                        DSPMCLib.vsiStatusGetDigitalOutput(port, pin, ref newPinState);
                        digitalOutputs[port - 11, pin].BackColor = newPinState ? Color.Lime : Color.Red;
                    }
                }

                for (int axis = 0; axis < 6; axis++)
                    DSPMCLib.vsiStatusGetAxisPosition(axis, ref axisPositions[axis]);

                EnableMotionControl(!axisMoving);

                XPos.Text = string.Format("{0:n4}", axisPositions[0]);
                YPos.Text = string.Format("{0:n4}", axisPositions[1]);
                ZPos.Text = string.Format("{0:n4}", axisPositions[2]);
                APos.Text = string.Format("{0:n4}", axisPositions[3]);
                BPos.Text = string.Format("{0:n4}", axisPositions[4]);
                CPos.Text = string.Format("{0:n4}", axisPositions[5]);

                isArmed = armedState;
            }
            else
            {
                DeactivateFormControls();
            }
        }


        private void btnConnect_Click(object sender, EventArgs e)
        {
            string serial = serialTxtBox.Text.Trim();

            if (DSPMCLib.vsiAPIConnect(serial, timer1.Interval, 2000) != DSPMCLib.VSI_ERROR_NONE)
            {
                string error = new string(new char[256]);
                int length = 0;
                DSPMCLib.vsiAPIGetLastNotification(ref error, ref length);
                MessageBox.Show("Connection Failed: " + error, "HiCON CSharp Demo");
                return;
            }
            else
                OnSuccessfulConnect();
        }


        private void btnARMPID_Click(object sender, EventArgs e)
        {
            if (isArmed)
                DSPMCLib.vsiCmdDisarm();
            else
                DSPMCLib.vsiCmdArm(AXIS_X0 | AXIS_Y1 | AXIS_Z2);
        }


        private void btnMove_Click(object sender, EventArgs e)
        {
            int Flags = 0;

            if (relativePositionBtn.Checked)
                Flags = DSPMCLib.VSI_MOVE_RELATIVE;
            else if (absolutePositionBtn.Checked)
                Flags = DSPMCLib.VSI_MOVE_ABSOLUTE;
            else if (velocityMotionBtn.Checked)
                Flags = DSPMCLib.VSI_MOVE_VELOCITY;

            if (DSPMCLib.vsiCmdExecuteMove(cbAxis.SelectedIndex,
                Convert.ToDouble(txtPosition.Text),
                Convert.ToDouble(txtVelocity.Text),
                Convert.ToDouble(txtAccel.Text),
                Flags,
                ++MoveSeqID[cbAxis.SelectedIndex]) != DSPMCLib.VSI_ERROR_NONE)
            {
                DisplayLastError();
            }
        }

        private void zeroMoveBtn_Click(object sender, EventArgs e)
        {
            int Flags = 0;

            Flags = Flags | DSPMCLib.VSI_MOVE_ABSOLUTE;

            for (int axis = 0; axis < MAX_AXIS; axis++)
            {
                foreach (Control control in axisGroup.Controls)
                {
                    if (control.Tag != null && (int)control.Tag == axis)
                    {
                        Label label = (Label)control;
                        if (Math.Abs(Convert.ToSingle(label.Text)) < 0.0001f)
                            goto SKIP;
                    }
                }

                if (DSPMCLib.vsiCmdExecuteMove(cbAxis.SelectedIndex,
                    0,
                    Convert.ToDouble(txtVelocity.Text),
                    Convert.ToDouble(txtAccel.Text),
                    Flags,
                    ++MoveSeqID[cbAxis.SelectedIndex]) != DSPMCLib.VSI_ERROR_NONE)
                {
                    DisplayLastError();
                }

            SKIP:
                continue;
            }
        }

        private void btnCancelMove_Click(object sender, EventArgs e)
        {
            if (DSPMCLib.vsiCmdCancelMove(cbAxis.SelectedIndex, false) != DSPMCLib.VSI_ERROR_NONE)
                DisplayLastError();
        }

        private void cancelAllBtn_Click(object sender, EventArgs e)
        {
            for (int axis = 0; axis < MAX_AXIS; axis++)
            {
                if (DSPMCLib.vsiCmdCancelMove(axis, false) != DSPMCLib.VSI_ERROR_NONE)
                    DisplayLastError();
            }
        }

        private void btnClearAxisPosition_Click(object sender, EventArgs e)
        {
            if (DSPMCLib.vsiCmdClearAxisPosition(cbAxis.SelectedIndex) != DSPMCLib.VSI_ERROR_NONE)
                DisplayLastError();
        }



        private void btnHome_Click(object sender, EventArgs e)
        {
            if (DSPMCLib.vsiCmdExecuteHoming(cbAxis.SelectedIndex, 0, 30.0, 7.5, true) != DSPMCLib.VSI_ERROR_NONE)
                DisplayLastError();
        }


        private void velocityChkBox_CheckedChanged(object sender, EventArgs e)
        {

        }


        private void jogLeftButton_MouseDown(object sender, MouseEventArgs e)
        {
            DSPMCLib.vsiCmdExecuteMove(cbAxis.SelectedIndex,
                -9999,
                Convert.ToDouble(txtVelocity.Text),
                Convert.ToDouble(txtAccel.Text),
                DSPMCLib.VSI_MOVE_ABSOLUTE,
                99);
        }


        private void jogLeftButton_MouseUp(object sender, MouseEventArgs e)
        {
            if (DSPMCLib.vsiCmdCancelMove(cbAxis.SelectedIndex, false) != DSPMCLib.VSI_ERROR_NONE)
                DisplayLastError();
        }


        private void jogRightButton_MouseDown(object sender, MouseEventArgs e)
        {
            DSPMCLib.vsiCmdExecuteMove(cbAxis.SelectedIndex,
                9999,
                Convert.ToDouble(txtVelocity.Text),
                Convert.ToDouble(txtAccel.Text),
                DSPMCLib.VSI_MOVE_ABSOLUTE,
                99);
        }


        private void jogRightButton_MouseUp(object sender, MouseEventArgs e)
        {
            if (DSPMCLib.vsiCmdCancelMove(cbAxis.SelectedIndex, false) != DSPMCLib.VSI_ERROR_NONE)
                DisplayLastError();
        }


        private void setVoltsSlider_ValueChanged(object sender, EventArgs e)
        {
            currentDACVolts[dacChannelCombobox.SelectedIndex] = (double)setVoltsSlider.Value / 100;
            setVoltsLabel.Text = currentDACVolts[dacChannelCombobox.SelectedIndex].ToString();

            if (DSPMCLib.vsiCmdSetDACOutput(dacChannelCombobox.SelectedIndex, currentDACVolts[dacChannelCombobox.SelectedIndex]) != DSPMCLib.VSI_ERROR_NONE)
                DisplayLastError();

            
        }

        private void dacChannelCombobox_SelectedIndexChanged(object sender, EventArgs e)
        {
            setVoltsLabel.Text = currentDACVolts[dacChannelCombobox.SelectedIndex].ToString();
            setVoltsSlider.Value = (int)(currentDACVolts[dacChannelCombobox.SelectedIndex] * 100);
        }

        private void zeroDACVoltsButton_Click(object sender, EventArgs e)
        {
            currentDACVolts[dacChannelCombobox.SelectedIndex] = 0;
            setVoltsSlider.Value = (int)(currentDACVolts[dacChannelCombobox.SelectedIndex] * 100);
            setVoltsLabel.Text = currentDACVolts[dacChannelCombobox.SelectedIndex].ToString();

            if (DSPMCLib.vsiCmdSetDACOutput(dacChannelCombobox.SelectedIndex, currentDACVolts[dacChannelCombobox.SelectedIndex]) != DSPMCLib.VSI_ERROR_NONE)
                DisplayLastError();

            int vectorCount = 1000;
            double[,] motionBuffer = new double[vectorCount, 8];

            for (int count = 0; count < vectorCount; count++)
                for(int axis = 0; axis < 3; axis++)
                    motionBuffer[count, axis] += 0.01;

            DSPMCLib.vsiCmdLoadBufferedMotion(motionBuffer, vectorCount);
        }







    }
}


