﻿using System;
using System.Net.Sockets;
using System.Runtime.InteropServices;
using System.Net;
using System.Windows.Forms;

namespace Smart3GComms
{
        public class Smart3GMsgPort
        {
            Socket mMsgSock;

            IPEndPoint LocalEndPoint;

            public Smart3GMsgPort( String LocalIP )
            {
                //local endpoint
                
                LocalEndPoint = new IPEndPoint(IPAddress.Parse(LocalIP), 44818);

                mMsgSock = new Socket(AddressFamily.InterNetwork,
                    SocketType.Dgram,
                    ProtocolType.Udp);

                // Binding is required with ReceiveFrom calls.
                mMsgSock.Bind(LocalEndPoint);
            }	
      
            /// <summary> 
            /// Send an explicit message to 3G card. writes the data to CW:25 ... CW:30
            /// </summary>
            public void SendExplicitMsg(byte IPdigit4, byte[] cwData_25_30)
            {
                if (cwData_25_30.Length < 6)
                    return;

                
                byte[] address = LocalEndPoint.Address.GetAddressBytes();
                address[3] = IPdigit4;
                IPAddress remIP = new IPAddress(address);

                IPEndPoint RemoteEndPoint = new IPEndPoint(remIP, 44818);

                LDRMsgENet TxStruct = new LDRMsgENet();
                TxStruct.LdrMsgSign = 0xDABA;
                TxStruct.SessionID = 0x9A8A7B6B;
                TxStruct.Len = 8;
                TxStruct.LdrData = new byte[8];
                TxStruct.LdrData[0]= IPdigit4; // last  byte of smart 3g card's IP
                TxStruct.LdrData[1] = 67;  //DNT_USER_MSG_CMD
            
                for(int i=2; i<8 ;i++)               // Msg data
                    TxStruct.LdrData[i] = cwData_25_30[i - 2];
                
                byte[] msg = StructureToByteArray(TxStruct);

                mMsgSock.SendTo(msg, msg.Length, SocketFlags.None, RemoteEndPoint);
             
            }
            
            static byte[] StructureToByteArray(object obj)
            {
                int len = Marshal.SizeOf(obj);
                byte[] arr = new byte[len];
                IntPtr ptr = Marshal.AllocHGlobal(len);
                Marshal.StructureToPtr(obj, ptr, true);
                Marshal.Copy(ptr, arr, 0, len);
                Marshal.FreeHGlobal(ptr);
                return arr;
            }

            /// <summary> 
            /// returns the last received explicit msg (6 bytes) from Smart3G (CW:17 ... CW:22)
            /// </summary>
            public bool GetRcvExplicitMsg(byte IPdigit4, ref byte[] CW_17_22)
            {
                if(ListExpMsgs[IPdigit4].DataAvaialble){

                    CW_17_22 = ListExpMsgs[IPdigit4].CW17_22;
                    ListExpMsgs[IPdigit4].DataAvaialble = false;
                    return true;
                }
                else
                return false;
            }

            
            /// <summary> 
            /// Buffer serial data packets from Smrt3G. Non blocking. When data packets are complete, returns number of bytes of the complete serial data.
            /// </summary>
            public bool GetRcvSerialData(byte IPdigit4, ref String SerialData )
            {
                if (ListSerialMsgs[IPdigit4].DataAvaialble)
                {
                    SerialData = ListSerialMsgs[IPdigit4].SerialData;
                    ListSerialMsgs[IPdigit4].DataAvaialble = false;
                    ListSerialMsgs[IPdigit4].SerialData = "";
                    return true;
                }
                else
                    return false;
            }


            [StructLayout(LayoutKind.Sequential, Pack = 1)]
            struct LDRMsgENet
            {
                public UInt16 LdrMsgSign;    // 0xDABA 				       
                public UInt16 Len;      	  // Ldr Msg Size 				
                public UInt32 SessionID;     // Session identifier 0x9A8A7B6B  

                [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
                public byte[] LdrData;

            }

            static void ByteArrayToStructure(byte[] bytearray, ref LDRMsgENet obj)
            {
                int len = Marshal.SizeOf(obj);
                IntPtr i = Marshal.AllocHGlobal(len);
                Marshal.Copy(bytearray, 0, i, len);
                obj = (LDRMsgENet)Marshal.PtrToStructure(i, obj.GetType());
                Marshal.FreeHGlobal(i);
            }

            struct ExpMsg
            {

                public byte[] CW17_22;
                public bool DataAvaialble;
            }

            struct SerialMsg
            {
               
                public String SerialData;
                public bool DataAvaialble;
            }

            ExpMsg[] ListExpMsgs = new ExpMsg[256];
            SerialMsg[] ListSerialMsgs = new SerialMsg[256];

            public void DoIOExchange()
            {
                LDRMsgENet LdrENetMsg = new LDRMsgENet();
                byte[] RxBytes = new byte[100];

                EndPoint DevAddr = new IPEndPoint(0, 0);

                if (mMsgSock.Available > 0)
                {
                    try
                    {
                        mMsgSock.ReceiveFrom(RxBytes, SocketFlags.None, ref DevAddr);
                    }
                    catch (SocketException e)
                    {
                        //Console.WriteLine("Source : " + e.Source);
                        //Console.WriteLine("Message : " + e.Message);
                    }

                    ByteArrayToStructure(RxBytes, ref LdrENetMsg);

                    if (LdrENetMsg.LdrMsgSign == 0x0000DABA && LdrENetMsg.SessionID == 0x9A8A7B6B && LdrENetMsg.Len >= 2)
                    {
                        IPEndPoint rcvaddr = (IPEndPoint)DevAddr;
                        byte[] addrbytes = rcvaddr.Address.GetAddressBytes();
                        int deviceId = addrbytes[3];

                        if (LdrENetMsg.LdrData[1] == 67) // Explicit msg
                        {
                        ListExpMsgs[deviceId].CW17_22 = new byte[6];
                            ListExpMsgs[deviceId].CW17_22[0] = LdrENetMsg.LdrData[2];
                            ListExpMsgs[deviceId].CW17_22[1] = LdrENetMsg.LdrData[3];
                            ListExpMsgs[deviceId].CW17_22[2] = LdrENetMsg.LdrData[4];
                            ListExpMsgs[deviceId].CW17_22[3] = LdrENetMsg.LdrData[5];
                            ListExpMsgs[deviceId].CW17_22[4] = LdrENetMsg.LdrData[6];
                            ListExpMsgs[deviceId].CW17_22[5] = LdrENetMsg.LdrData[7];
                            ListExpMsgs[deviceId].DataAvaialble = true;
                        }
                        else if (LdrENetMsg.LdrData[1] == 65) // Barcode/Serial Data
                        {
                            for(int count = 0; count <LdrENetMsg.Len-2;count++)
                                ListSerialMsgs[deviceId].SerialData += Convert.ToChar(LdrENetMsg.LdrData[count + 2]);
                        }
                        else if (LdrENetMsg.LdrData[1] == 66)
                        {
                            for (int count = 0; count < LdrENetMsg.Len - 2; count++)
                                ListSerialMsgs[deviceId].SerialData += Convert.ToChar(LdrENetMsg.LdrData[count + 2]);
                            ListSerialMsgs[deviceId].DataAvaialble = true;
                        }
                    }
                }
            }
        }
    
    public class Smart3GPoller
    {
        /// <summary> 
        /// Poll Intervals are in millisecond.  TimeoutMultiplier is used to shift left 4 and multiply the result with PollInterval to calculate connection timeout value.
        /// e.g., for TimeoutMultiplier of 3 and PollInterval of 100: 4 shl 3 x 100 give timeout as 3200 ms or 3.2 seconds
        /// </summary>
        public Smart3GPoller(String LocalIP, byte DeviceNodeID, uint PC_To_3G_PollInterval, uint _3G_To_PC_PollInterval, uint TimeoutMultiplier)
        {
            mTxRPI = PC_To_3G_PollInterval;  //millisec
            mRxRPI = _3G_To_PC_PollInterval;  //millisec
            mTimeoutMult = TimeoutMultiplier;  //4 << mult.  0..13 only

            IPAddress LIP = IPAddress.Parse(LocalIP);
            byte [] LIPBytes = LIP.GetAddressBytes();

            LIPBytes[3] = DeviceNodeID;
            NodeID = DeviceNodeID;

            IPAddress RIP = new IPAddress(LIPBytes);
            mIpAddr = RIP.ToString();

            RemoteEndPoint = new IPEndPoint(IPAddress.Parse(mIpAddr), 44818);

            mSock = new Socket(AddressFamily.InterNetwork,
                SocketType.Dgram,
                ProtocolType.Udp);

            //local endpoint
            IPEndPoint LocalEndPoint = new IPEndPoint(IPAddress.Parse(LocalIP), 0);

            // Binding is required with ReceiveFrom calls.
            mSock.Bind(LocalEndPoint);
            
            mTxScanData = new byte[4];
            mRxScanData = new byte[4];
        }

        ~Smart3GPoller()
        {
            mSock.Close();
        }

        public byte NodeID;
    
        /// <summary> 
        /// Control Words 1 and 2 data from Smar3G Device
        /// </summary>
        public UInt16 mCW1, mCW2;

        /// <summary> 
        /// Read CW1 Bit Status, BitPosition: 0..15
        /// </summary>
        public bool ReadCW1_Bit(int BitPosition) { return ((mCW1 & (1 << BitPosition)) > 0) ? true : false; }

        /// <summary> 
        /// Read CW2 Bit Status, BitPosition: 0..15
        /// </summary>
        public bool ReadCW2_Bit(int BitPosition) { return ((mCW2 & (1 << BitPosition)) > 0) ? true : false; }

        /// <summary> 
        /// Comms Status with Smar3G Device
        /// </summary>
        public bool mDeviceOnline;

        /// <summary> 
        /// Read Input Status for all 8 pins
        /// </summary>
        public Byte ReadInputs() { return mRxScanData[0]; }

        /// <summary> 
        /// Read Output Status for all 8 pins
        /// </summary>
        public Byte ReadOutputs() { return mRxScanData[2]; }

        /// <summary> 
        /// Read Input Status, pin: 1..8
        /// </summary>
        public bool ReadInputPin(int pin) { return ((mRxScanData[0] & (1<<(pin-1)))>0)?true:false; }

        /// <summary> 
        /// Read Output Status, pin: 1..8
        /// </summary>
        public bool ReadOutputPin(int pin) { return ((mRxScanData[2] & (1 << (pin - 1))) > 0) ? true : false; }

        /// <summary> 
        /// Write to any Output.  pin = 1..8, val = true or false. Network I/O check box should be on in Smart3G configuration.
        /// Data is sent to Smart3G on the next tx poll.
        /// </summary>
        public void WriteOutputPin(bool val, int pin)
        {
            if (val)
            {
                mTxScanData[0] = (byte)(mTxScanData[0] | (byte)(1 << (pin - 1)));
            }
            else
            {
                mTxScanData[0] = (byte)(mTxScanData[0] & (byte)~(1 << (pin - 1)));
            }
        }

        /// <summary> 
        /// Write to selected bit(0..15) in CW:9.  Network I/O check box should be off in Smart3G configuration.
        /// Data is sent to Smart3G on the next tx poll.
        /// </summary>
        public void WriteCW9Bit(int BitPosition, bool BitVal)
        {
            if( BitPosition < 8 )
                mTxScanData[0] = BitVal? (byte)(mTxScanData[0] | (byte)(1 << BitPosition)) : (byte)(mTxScanData[0] & (byte)~(1 << BitPosition));
            else if (BitPosition < 16)
                mTxScanData[1] = BitVal? (byte)(mTxScanData[1] | (byte)(1 << (BitPosition-8))) : (byte)(mTxScanData[1] & (byte)~(1 << (BitPosition-8)));
        }

        /// <summary> 
        /// Write to selected bit(0..15) in CW:10.  Network I/O check box should be off in Smart3G configuration.
        /// Data is sent to Smart3G on the next tx poll.
        /// </summary>
        public void WriteCW10Bit(int BitPosition, bool BitVal)
        {
            if (BitPosition < 8)
                mTxScanData[2] = BitVal ? (byte)(mTxScanData[2] | (byte)(1 << BitPosition)) : (byte)(mTxScanData[2] & (byte)~(1 << BitPosition));
            else if (BitPosition < 16)
                mTxScanData[3] = BitVal ? (byte)(mTxScanData[3] | (byte)(1 << (BitPosition - 8))) : (byte)(mTxScanData[3] & (byte)~(1 << (BitPosition - 8)));
        }

        /// <summary> 
        /// Write 16 bit data to CW:9.  Network I/O check box should be off in Smart3G configuration.
        /// Data is sent to Smart3G on the next tx poll.
        /// </summary>
        public void WriteCW9(UInt16 CW9Data)
        {
            mTxScanData[0] = (byte)CW9Data;
            mTxScanData[1] = (byte)(CW9Data >> 8);
        }

        /// <summary> 
        /// Write 16 bit data to CW:10. Network I/O check box should be off in Smart3G configuration.
        /// Data is sent to Smart3G on the next tx poll.
        /// </summary>
        public void WriteCW10(UInt16 CW10Data)
        {
            mTxScanData[2] = (byte)CW10Data;
            mTxScanData[3] = (byte)(CW10Data >> 8);
        }

        /// <summary> 
        /// Execute Comms Protocol with Smart3G.  must be run through a timer with 1 millisec period or faster.
        /// </summary>
        public void DoIOExchange()
        {
            //Device IP address not initialized yet
            if (mIpAddr.Length == 0)
                return;

            _3GiodataRx RxStruct = new _3GiodataRx();

            //empty message buffer
            byte[] RxDat = new byte[Marshal.SizeOf(RxStruct)];
            EndPoint DevAddr = new IPEndPoint(0,0);
            IPEndPoint SrcIPAddr;

            while (mSock.Available > 0)
            {
                try
                {
                    mSock.ReceiveFrom(RxDat, SocketFlags.None, ref DevAddr);
                }
                catch (SocketException e)
                {
                    //Console.WriteLine("Source : " + e.Source);
                    //Console.WriteLine("Message : " + e.Message);
                }

                SrcIPAddr = (IPEndPoint)DevAddr;

                ByteArrayToStructure(RxDat, ref RxStruct);

                if (RxStruct.MsgSign == 0x80020002 && SrcIPAddr.Address.ToString() == mIpAddr)
                {
                    mRxScanData[0] = RxStruct.RxIOdata[0]; //inputs
                    mRxScanData[1] = RxStruct.RxIOdata[1]; //i/o enable bit
                    mRxScanData[2] = RxStruct.RxIOdata[2]; //outputs
                    mRxScanData[3] = RxStruct.RxIOdata[3];

                    mCW1 = RxStruct.CW1;
                    mCW2 = RxStruct.CW2;
                    
                    mRxTimer = (uint)System.Environment.TickCount + (mRxRPI * (uint)(4<< (UInt16)mTimeoutMult));	//calc Rx timeout

                    if (!mDeviceOnline)
                        mTxTimer = 0;

                    mDeviceOnline = true;
                }
            }

            if (mDeviceOnline)
            {
                if (System.Environment.TickCount > mTxTimer) 	//time to transmit
                {
                    _3GiodataTx TxBfr = new _3GiodataTx();

                    TxBfr.TxOutdata = BitConverter.ToUInt32(mTxScanData, 0);

                    byte[] msg = StructureToByteArray(TxBfr);

                    RemoteEndPoint.Port = 2222;
                    mSock.SendTo(msg, msg.Length, SocketFlags.None, RemoteEndPoint);

                    mTxTimer = (uint)System.Environment.TickCount + mTxRPI; //calc time for next transmission
                }

                if ((uint)System.Environment.TickCount > mRxTimer) //timed out waiting for Rx data from Device
                {
                    //no response from device.  mark offline
                    mDeviceOnline = false;
                    mTxTimer = 0;
                    //mLastRxTime = -1;
                    //clear input scan memory
                    mRxScanData[0] = 0;
                    mRxScanData[1] = 0;
                    mRxScanData[2] = 0;
                    mRxScanData[3] = 0;
                    mCW1 = 0;
                    mCW2 = 0;
                }

            }
            else //Device is offline
            {
                if ((int)System.Environment.TickCount > mTxTimer)	//time to transmit connect cmd
                {
                    SMART3G_IOCmdMsg Smart3GCmd = new SMART3G_IOCmdMsg();

                    Smart3GCmd.iCmd = 0x004C006F;
                    Smart3GCmd.iCmdZ = 2;
                    Smart3GCmd.iCmdU = 0x003C00B2;
                    Smart3GCmd.Cmdx1 = 0x06200254;
                    Smart3GCmd.Cmdx2 = 0x0124;
                    Smart3GCmd.vid = 0x380;
                    Smart3GCmd.TimeoutMult = mTimeoutMult;
                    Smart3GCmd.Host_RPI = mTxRPI * 1000;
                    Smart3GCmd.Smart3G_RPI = mRxRPI * 1000;

                    byte[] msg = StructureToByteArray(Smart3GCmd);

                    RemoteEndPoint.Port = 44818;
                    mSock.SendTo(msg, msg.Length, SocketFlags.None, RemoteEndPoint);
                    
                    mTxTimer = (uint)System.Environment.TickCount + 350; //calc time for re-transmission
                }
            }
        }

        IPEndPoint RemoteEndPoint;
        Socket mSock;
        UInt32 mTxTimer, mRxTimer;
        byte[] mTxScanData;
        byte[] mRxScanData;
        string mIpAddr;
        uint mTxRPI, mRxRPI, mTimeoutMult;

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        struct SMART3G_IOCmdMsg
        {
            public UInt32 iCmd;
            UInt32 Rsv1w1, Rsv1w2, Rsv1w3, Rsv1w4, Rsv1w5, Rsv1w6;
            UInt16 Rsv2;

            public UInt16 iCmdZ;
            UInt32 Rsv3;
            public UInt32 iCmdU;

            public UInt32 Cmdx1;
            public UInt16 Cmdx2;

            UInt32 Rsv4w1, Rsv4w2, Rsv4w3;
            public UInt16 vid;

            UInt32 Rsv51; //sn
            public UInt32 TimeoutMult; //Timeout Multiplier 0..7 (4 << mul)
            public UInt32 Host_RPI;
            UInt16 Rsv6;

            public UInt32 Smart3G_RPI;
            UInt16 Rsv7w1, Rsv7w2;
        };

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        struct _3GiodataRx
        {
            public UInt32 MsgSign;  //0x80020002
            UInt32 Rsvd1w1, Rsvd1w2, Rsvd1w3, Rsvd1w4, Rsvd1w5;

            //[MarshalAs(UnmanagedType.U1, SizeConst = 4)]

            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
            public byte[] RxIOdata;   //Byte 0 = Inputs, Byte 1 = Output Enable Bit, Byte 2 = Outputs,  

            public UInt16 CW1;	  //Ladder Control Word 1 and 2
            public UInt16 CW2;
        }

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        struct _3GiodataTx
        {
            UInt32 Rsvd1w1, Rsvd1w2, Rsvd1w3, Rsvd1w4, Rsvd1w5, Rsvd1w6;
            public UInt32 TxOutdata;	//Byte 0 maps to Digital Outputs if VirtualIO is on.  
                                        //Byte 0..3 maps to Ladder Control Word 9 and 10 if VirtualIO is off
        };

        static byte[] StructureToByteArray(object obj)
        {
            int len = Marshal.SizeOf(obj);
            byte[] arr = new byte[len];
            IntPtr ptr = Marshal.AllocHGlobal(len);
            Marshal.StructureToPtr(obj, ptr, true);
            Marshal.Copy(ptr, arr, 0, len);
            Marshal.FreeHGlobal(ptr);
            return arr;
        }

        static void ByteArrayToStructure(byte[] bytearray, ref _3GiodataRx obj)
        {
            int len = Marshal.SizeOf(obj);
            IntPtr i = Marshal.AllocHGlobal(len);
            Marshal.Copy(bytearray, 0, i, len);
            obj = (_3GiodataRx)Marshal.PtrToStructure(i, obj.GetType());
            Marshal.FreeHGlobal(i);
        }
    }
}
 


    /*

public static object RawDeserialize( byte[] rawData, int position, Type anyType )
{
int rawsize = Marshal.SizeOf( anyType );
if( rawsize > rawData.Length )
return null;
IntPtr buffer = Marshal.AllocHGlobal( rawsize );
Marshal.Copy( rawData, position, buffer, rawsize );
object retobj = Marshal.PtrToStructure( buffer, anyType );
Marshal.FreeHGlobal( buffer );
return retobj;
}

the position is the position into the byte array to start deserializing
from, and the
type is the type of the structure - typof(MESSAGE_LOG_HEADER_STRUCT2).
Remember to cast the return value to the same type too.

And if you want to do vice versa, use this one:

public static byte[] RawSerialize( object anything )
{
int rawSize = Marshal.SizeOf( anything );
IntPtr buffer = Marshal.AllocHGlobal( rawSize );
Marshal.StructureToPtr( anything, buffer, false );
byte[] rawDatas = new byte[ rawSize ];
Marshal.Copy( buffer, rawDatas, 0, rawSize );
Marshal.FreeHGlobal( buffer );
return rawDatas;
}
*/
